/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor.reference
{

    using System;
    using System.Web;
    using org.owasp.appsensor;
    using org.owasp.appsensor.reference.adapters;
    using Owasp.Esapi;

    /**
     * This class is the default implementation of the utilities helper
     * class for AppSensor.  This interface is responsible for providing 
     * the implemementation classes for a number of utility type objects 
     * that are needed throughout AppSensor.
     * <p />  
     * Since AppSensor is based on and closely tied to Esapi, the default
     * implementation simply uses direct Esapi classes / methods or 
     * adapters that use Esapi implementations under the covers.  
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since May 6, 2010
     */
    public class DefaultASUtilities : ASUtilities
    {

        private static volatile ASUtilities singletonInstance;

        public static ASUtilities GetInstance()
        {
            if (singletonInstance == null)
            {
                lock (typeof(DefaultASUtilities))
                {
                    if (singletonInstance == null)
                    {
                        singletonInstance = new DefaultASUtilities();
                    }
                }
            }
            return singletonInstance;
        }

        /**
         * This method uses the Esapi current user implementation
         * @return ESAPIASCurrentUser adapter class that uses Esapi's 
         * 		current user implementation
         */
        public ASUser CurrentUser
        {
            get
            {
                return new ESAPIASCurrentUser();
            }
        }

        /**
         * This method uses the Esapi logger implementation
         * @return ESAPIASLogger adapter class that uses Esapi's 
         * 		logger implementation
         */
        public ASLogger GetLogger(String className)
        {
            return new ESAPIASLogger(className);
        }

        /**
         * This method uses the Esapi current request implementation
         * @return current http request retrieved from Esapi
         */
        public HttpRequest GetCurrentRequest()
        {
            if (HttpContext.Current != null)
            {
                return HttpContext.Current.Request;
            }
            return null;
        }

    }
}